"""
distinction.py

Implementation of the distinction (trade) operator X for the tick‑flip algebra.

The distinction operator reflects a tick state across the central layer,
effectively swapping inward and outward contexts.  In array terms this
corresponds to reversing the distribution.
"""

from .tick_state import TickState


def distinction(state: TickState) -> TickState:
    """Reflect a tick state across the origin (r ↦ −r).

    Parameters
    ----------
    state : TickState
        The input tick state.  Its distribution is not mutated.

    Returns
    -------
    TickState
        A new tick state whose distribution is the reverse of the input.
    """
    # Reverse the distribution array (equivalent to r -> -r)
    new_dist = state.distribution[::-1].copy()
    return TickState(new_dist, state.N)


def X(state: TickState) -> TickState:
    """
    Alias for the distinction operator.  Provided for convenience.

    Parameters
    ----------
    state : TickState
        The input tick state.

    Returns
    -------
    TickState
        The result of applying the distinction operator to ``state``.
    """
    return distinction(state)
